/** @file   LightBeam.cpp
 * @brief   Implementation of LightBeam class.
 * @version $Revision: 1.1 $
 * @date    $Date: 2006/06/17 21:49:11 $
 * @author  Tomi Lamminsaari
 */

#include "LightBeam.h"
#include "LightBeamCell.h"
#include "www_assert.h"
#include "www_map.h"

using eng2d::Vec2D;

namespace WeWantWar {

LightBeam::LightBeam( int aCellSize) :
  iPosition(0,0),
  iDirection(0),
  iCellSize(aCellSize),
  iBackBuffer1(0),
  iBackBuffer2(0)
{
}

LightBeam::~LightBeam()
{
  for ( int i=0; i < iLightBeamCells.size(); i++ ) {
    delete iLightBeamCells.at(i);
  }
  iLightBeamCells.clear();
  
  if ( iBackBuffer1 != 0 ) {
    destroy_bitmap( iBackBuffer1 );
  }
  if ( iBackBuffer2 != 0 ) {
    destroy_bitmap( iBackBuffer2 );
  }
}

bool LightBeam::mapCollision( const Vec2D& aPosition, int aLayer ) const
{
  Map::selectLayer( Map::int2LayerId(aLayer) );
  return Map::collide( aPosition );
}

bool LightBeam::bulletCollision( const Vec2D& aPosition ) const
{
  return Map::bulletCollide( aPosition );
}

int LightBeam::constructFromBitmap(BITMAP* aBitmap )
{
  WWW_ASSERT( aBitmap != 0 );
  int cx = aBitmap->w / 2;
  int cy = aBitmap->h / 2;
  
  int htiles = aBitmap->w / iCellSize;
  int vtiles = aBitmap->h / iCellSize;
  
  for ( int vx=0; vx < htiles/2; vx++ ) {
    for ( int vy=0; vy < vtiles/2; vy++ ) {
      int x1 = cx + ( vx*iCellSize );
      int x2 = cx - ( vx*iCellSize );
      int y1 = cy + ( vy*iCellSize );
      int y2 = cy - ( vy*iCellSize );
      
      int c1 = getpixel(aBitmap, x1,y1);
      int c2 = getpixel(aBitmap, x1,y2);
      int c3 = getpixel(aBitmap, x2,y1);
      int c4 = getpixel(aBitmap, x2,y2);
      
      int col1 = getr( getpixel(aBitmap, x1,y1) );
      int col2 = getr( getpixel(aBitmap, x1,y2) );
      int col3 = getr( getpixel(aBitmap, x2,y1) );
      int col4 = getr( getpixel(aBitmap, x2,y2) );
      
      if ( c1 != 0 ) {
        addLightBeamCell( new LightBeamCell( Vec2D(x1-cx,y1-cy), col1, iCellSize, c1 ) );
      }
      if ( c2 != 0 ) {
        addLightBeamCell( new LightBeamCell( Vec2D(x1-cx,y2-cy), col2, iCellSize, c2 ) );
      }
      if ( c3 != 0 ) {
        addLightBeamCell( new LightBeamCell( Vec2D(x2-cx,y1-cy), col3, iCellSize, c3 ) );
      }
      if ( c4 != 0 ) {
        addLightBeamCell( new LightBeamCell( Vec2D(x2-cx,y2-cy), col4, iCellSize, c4 ) );
      } 
    }
  }
  
  iBackBuffer1 = create_bitmap( aBitmap->w, aBitmap->h );
  iBackBuffer2 = create_bitmap( aBitmap->w, aBitmap->h );
  if ( iBackBuffer1 == 0 || iBackBuffer2 == 0 ) {
    return -1;
  }
  return 0;
}

void LightBeam::setBeamPosition( const Vec2D& aPosition, float aDirection )
{
  iPosition = aPosition;
  iDirection = aDirection;
}

void LightBeam::update()
{
}

void LightBeam::applyLightBeam( BITMAP* aCanvas, const Vec2D& aPos )
{
//  int destx = iPosition.intX() - Map::scrollX;
//  int desty = iPosition.intY() - Map::scrollY;
  this->renderLightBeamToBuffer();
  int destx = aPos.intX();
  int desty = aPos.intY();
  destx -= iBackBuffer2->w/2;
  desty -= iBackBuffer2->h/2;
  
  set_multiply_blender( 0,0,0, 255 );
  draw_trans_sprite( aCanvas, iBackBuffer2, destx, desty );
}

void LightBeam::addLightBeamCell( LightBeamCell* aCell )
{
  if ( aCell != 0 ) {
    LOG_MESSAGE( "*** New light cell ***" );
    std::string foo = "x:";
    foo += eng2d::Int2String( static_cast<int>(aCell->cellPosition().intX()) );
    foo += ", y:";
    foo += eng2d::Int2String( static_cast<int>(aCell->cellPosition().intY()) );
    LOG_MESSAGE( foo.c_str() );
    iLightBeamCells.push_back( aCell );
  }
}

void LightBeam::renderLightBeamToBuffer()
{
  for ( int i=0; i < iLightBeamCells.size(); i++ ) {
    iLightBeamCells.at(i)->update( iPosition, *this, iDirection );
  }
  
  clear_to_color( iBackBuffer1, makecol(255,0,255) );
  clear_bitmap( iBackBuffer2 );
  Vec2D origo( iBackBuffer1->w/2, iBackBuffer1->h/2 );
  
  textprintf_ex(iBackBuffer1, font, 0,0, 0,-1, "%d", iLightBeamCells.size() );
  for ( int i=0; i < iLightBeamCells.size(); i++ ) {
    iLightBeamCells.at(i)->renderBrightness( iBackBuffer1, origo );
  }
  rotate_sprite( iBackBuffer2, iBackBuffer1, 0,0, ftofix(iDirection) );
}

void LightBeam::lightBeamSize( int& aWidth, int& aHeight ) const
{
  if ( iBackBuffer2 != 0 ) {
    aWidth = iBackBuffer2->w;
    aHeight = iBackBuffer2->h;
  } else {
    aWidth = 0;
    aHeight = 0;
  }
}

} // end of namespace
